# Copyright 2024 Bytedance Ltd. and/or its affiliates
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


from collections import defaultdict
from typing import Any, Callable, Dict, Tuple, TypedDict

import torch
from transformers import PreTrainedTokenizer

from ...protocol import DataProto
from ...utils.reward_score import math_compute_score, r1v_compute_score, r1gui_compute_score

##################################################################################
from .verify import ground_only_positive,load_verify,load_tokenizer,load_processor
##################################################################################


import torch
import numpy as np
from PIL import Image



##################################################################################
import re
from typing import List, Tuple, Union

Number = Union[int, float]

def extract_points(text: str) -> List[Tuple[Number, Number]]:
    # 匹配 'point': [x, y] 或 "point": [x, y]
    num = r"[+-]?(?:\d+(?:\.\d*)?|\.\d+)(?:[eE][+-]?\d+)?"
    pattern = re.compile(rf"""['"]point['"]\s*:\s*\[\s*({num})\s*,\s*({num})\s*\]""")
    pts: List[Tuple[Number, Number]] = []

    for x, y in pattern.findall(text):
        # 尽量转成 int，否则用 float
        def to_num(s: str) -> Number:
            return int(float(s)) if re.fullmatch(r"[+-]?\d+", s) else float(s)
        pts.append([to_num(x), to_num(y)])
    return pts

def point_in_bbox(point, bbox):
    if len(point) != 2 or len(bbox) != 4:
        raise ValueError("point must be [x, y] and bbox must be [x_min, y_min, x_max, y_max]")
    x, y = point
    x_min, y_min, x_max, y_max = bbox

    return (x_min <= x <= x_max) and (y_min <= y <= y_max)
##################################################################################






class RewardScore(TypedDict):
    overall: float
    format: float
    accuracy: float



import base64
from PIL import Image
import io
import json
import numpy as np


####################################################################
def bbox_to_center(bbox):
    bbox = np.array(bbox, dtype=float)
    if bbox.ndim == 1:
        bbox = bbox[None, :]  # (1, 4)

    if np.all(bbox[:, 2] > bbox[:, 0]) and np.all(bbox[:, 3] > bbox[:, 1]):
        # 形如 [x_min, y_min, x_max, y_max]
        cx = (bbox[:, 0] + bbox[:, 2]) / 2
        cy = (bbox[:, 1] + bbox[:, 3]) / 2
    else:
        # 形如 [x, y, w, h]
        cx = bbox[:, 0] + bbox[:, 2] / 2
        cy = bbox[:, 1] + bbox[:, 3] / 2

    centers = np.stack([cx, cy], axis=1)
    return centers
####################################################################


class CustomRewardManager:
    def __init__(self, tokenizer: PreTrainedTokenizer, compute_score: str):
        self.tokenizer = tokenizer
        if compute_score == "math":
            self.compute_score: Callable[[str, str], RewardScore] = math_compute_score
        elif compute_score == "r1v":
            self.compute_score: Callable[[str, str], RewardScore] = r1v_compute_score
        elif compute_score == "r1gui":
            self.compute_score: Callable[[str, str], RewardScore] = r1gui_compute_score
        else:
            raise NotImplementedError()
        
        ####################################################################################
        model_name_or_path = "/mnt/task_runtime/sh_workspace/new_temp/Models/GUI-Actor-Verifier-2B"
        self.verify_model = load_verify(model_name_or_path)
        self.tokenizer = load_tokenizer(model_name_or_path)
        self.processor = load_processor(model_name_or_path)
        ####################################################################################

    def __call__(self, data: DataProto) -> Tuple[torch.Tensor, Dict[str, Any]]:
        reward_tensor = torch.zeros_like(data.batch["responses"], dtype=torch.float32)
        reward_metrics = defaultdict(list)
        for i in range(len(data)):
            data_item = data[i]  # DataProtoItem
            response_ids = data_item.batch["responses"]
            response_mask = data_item.batch["response_mask"]
            valid_response_length = response_mask.sum()
            valid_response_ids = response_ids[:valid_response_length]

            response_str = self.tokenizer.decode(valid_response_ids, skip_special_tokens=True)
            ground_truth = data_item.non_tensor_batch["ground_truth"]

            ####################################################################################
            score_rule = self.compute_score(response_str, ground_truth)
            ####################################################################################
            
            
            ####################################################################################
            image_bytes = data_item.non_tensor_batch['image']['bytes']
            
            ############################################
            image_file = io.BytesIO(image_bytes)
            image = Image.open(image_file)
            # width, height = image.size
            instruction= data_item.non_tensor_batch['instruction']
            point = extract_points(response_str)[0]
            gt_dict = json.loads(ground_truth)
            gt_point = bbox_to_center(gt_dict['gt_bbox_ori'])
            print('Pred Point: ', point)
            print('GT : ', gt_dict['gt_bbox_ori'])
            print('GT Point : ', gt_point)
            
            _, probs = ground_only_positive(self.verify_model, self.tokenizer, self.processor, instruction, image, point)
            
            
            # 统一把 probs 变成 Python float
            if torch.is_tensor(probs):
                probs = probs.detach().float().cpu()
                probs = probs.mean().item() if probs.numel() > 1 else probs.item()
            else:
                probs = float(probs)
            ####################################################################################
            
            score = {
                "overall": 0.8 * probs + 0.2 * 1,
                "format": 1,
                "accuracy": probs,
            }
            
            ####################################################################################
            
            
            
            
            
            
            reward_tensor[i, valid_response_length - 1] = score["overall"]
            for key, value in score.items():
                reward_metrics[key].append(value)

        return reward_tensor, reward_metrics
