# Copyright 2024 Bytedance Ltd. and/or its affiliates
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


from collections import defaultdict
from typing import Any, Callable, Dict, Tuple, TypedDict

import torch
from transformers import PreTrainedTokenizer

from ...protocol import DataProto
from ...utils.reward_score import math_compute_score, r1v_compute_score, r1gui_compute_score

##################################################################################
from .verify import ground_only_positive,load_verify,load_tokenizer,load_processor
##################################################################################


import torch
import numpy as np
from PIL import Image



##################################################################################
import re
from typing import List, Tuple, Union

Number = Union[int, float]

def extract_points(text: str) -> List[Tuple[Number, Number]]:
    # 匹配 'point': [x, y] 或 "point": [x, y]
    num = r"[+-]?(?:\d+(?:\.\d*)?|\.\d+)(?:[eE][+-]?\d+)?"
    pattern = re.compile(rf"""['"]point['"]\s*:\s*\[\s*({num})\s*,\s*({num})\s*\]""")
    pts: List[Tuple[Number, Number]] = []

    for x, y in pattern.findall(text):
        # 尽量转成 int，否则用 float
        def to_num(s: str) -> Number:
            return int(float(s)) if re.fullmatch(r"[+-]?\d+", s) else float(s)
        pts.append([to_num(x), to_num(y)])
    return pts

def point_in_bbox(point, bbox):
    if len(point) != 2 or len(bbox) != 4:
        raise ValueError("point must be [x, y] and bbox must be [x_min, y_min, x_max, y_max]")
    x, y = point
    x_min, y_min, x_max, y_max = bbox

    return (x_min <= x <= x_max) and (y_min <= y <= y_max)
##################################################################################






class RewardScore(TypedDict):
    overall: float
    format: float
    accuracy: float



import base64
from PIL import Image
import io
import json
import numpy as np


####################################################################
def bbox_to_center(bbox):
    bbox = np.array(bbox, dtype=float)
    if bbox.ndim == 1:
        bbox = bbox[None, :]  # (1, 4)

    if np.all(bbox[:, 2] > bbox[:, 0]) and np.all(bbox[:, 3] > bbox[:, 1]):
        # 形如 [x_min, y_min, x_max, y_max]
        cx = (bbox[:, 0] + bbox[:, 2]) / 2
        cy = (bbox[:, 1] + bbox[:, 3]) / 2
    else:
        # 形如 [x, y, w, h]
        cx = bbox[:, 0] + bbox[:, 2] / 2
        cy = bbox[:, 1] + bbox[:, 3] / 2

    centers = np.stack([cx, cy], axis=1)
    return centers[0].tolist()

def fix_bbox_shape(lst):
    fixed = []
    for item in lst:
        if isinstance(item, (list, tuple)):
            if len(item) == 2:
                # 把 [x, y] 复制成 [x, y, x, y]
                x, y = item
                fixed.append([x, y, x, y])
            elif len(item) == 4:
                fixed.append(item)
            else:
                print(f"⚠️ 跳过长度异常项: {item}")
        else:
            print(f"⚠️ 非list/tuple项: {item}")
    return fixed
####################################################################


class CustomRewardManager:
    def __init__(self, tokenizer: PreTrainedTokenizer, compute_score: str):
        self.tokenizer = tokenizer
        if compute_score == "math":
            self.compute_score: Callable[[str, str], RewardScore] = math_compute_score
        elif compute_score == "r1v":
            self.compute_score: Callable[[str, str], RewardScore] = r1v_compute_score
        elif compute_score == "r1gui":
            self.compute_score: Callable[[str, str], RewardScore] = r1gui_compute_score
        else:
            raise NotImplementedError()
        
        ####################################################################################
        model_name_or_path = "/mnt/beegfs/Workspace/Models/GUI-Actor-Verifier-2B"
        self.verify_model = load_verify(model_name_or_path)
        self.tokenizer = load_tokenizer(model_name_or_path)
        self.processor = load_processor(model_name_or_path)
        ####################################################################################

    def __call__(self, data: DataProto) -> Tuple[torch.Tensor, Dict[str, Any]]:
        reward_tensor = torch.zeros_like(data.batch["responses"], dtype=torch.float32)
        reward_metrics = defaultdict(list)
        
        ############################################
        all_info_list_Pred_Point = []
        all_info_list_GT_BBox_Ori = []
        all_info_list_GT_Point = []
        all_info_list_Score = []
        
        # print(len(data)) # 2560 / 4*5*128
        ############################################
        
        for i in range(len(data)):
            data_item = data[i]  # DataProtoItem
            response_ids = data_item.batch["responses"]
            response_mask = data_item.batch["response_mask"]
            valid_response_length = response_mask.sum()
            valid_response_ids = response_ids[:valid_response_length]

            response_str = self.tokenizer.decode(valid_response_ids, skip_special_tokens=True)
            ground_truth = data_item.non_tensor_batch["ground_truth"]

            ####################################################################################
            score_rule = self.compute_score(response_str, ground_truth)
            ####################################################################################
            
            ####################################################################################
            image_bytes = data_item.non_tensor_batch['image']['bytes']
            ############################################
            image_file = io.BytesIO(image_bytes)
            image = Image.open(image_file)
            # width, height = image.size
            instruction= data_item.non_tensor_batch['instruction']
            
            gt_dict = json.loads(ground_truth)
            
            if len(gt_dict['gt_bbox_ori']) > 2:
                gt_point = bbox_to_center(gt_dict['gt_bbox_ori'])
            else: 
                gt_point = gt_dict['gt_bbox_ori']
                
            gt_scale_width = gt_dict['scale_width']
            gt_scale_height = gt_dict['scale_height']
            temp_xy = extract_points(response_str)
            
            if len(temp_xy) < 1:
                # temp_xy.append([-0.1,-0.1])
                point = [-0.1,-0.1]
            else: 
                point = temp_xy[0]
                point[0] = point[0] / gt_scale_width
                point[1] = point[1] / gt_scale_height
            
            
            
            ####################################################################################
            print('Pred_Point: ', point)
            print('GT_BBox_Ori: ', gt_dict['gt_bbox_ori'])
            print('GT_Point : ', gt_point)
            print('gt_scale_width: ', gt_scale_width)
            print('gt_scale_height: ', gt_scale_height)
            
            # all_info = {}
            # all_info['Pred_Point'] = point
            # all_info['GT_BBox_Ori'] = gt_dict['gt_bbox_ori']
            # all_info['GT_Point'] = gt_point
            # all_info['gt_scale_width'] = gt_scale_width
            # all_info['gt_scale_height'] = gt_scale_height
            
            all_info_list_Pred_Point.append(point)
            all_info_list_GT_BBox_Ori.append(gt_dict['gt_bbox_ori'])
            all_info_list_GT_Point.append(gt_point)
            ####################################################################################
            _, probs = ground_only_positive(self.verify_model, self.tokenizer, self.processor, instruction, image, point)
            print('------------------------------------------------------')
            
            
            # 统一把 probs 变成 Python float
            if torch.is_tensor(probs):
                probs = probs.detach().float().cpu()
                probs = probs.mean().item() if probs.numel() > 1 else probs.item()
            else:
                probs = float(probs)
            ####################################################################################
            
            score = {
                "overall": 1 * probs + 0.2 * 0,
                "format": 1,
                "accuracy": probs,
            }
            
            # all_info['score'] = probs
            all_info_list_Score.append(probs)
            ####################################################################################
            
            
            reward_tensor[i, valid_response_length - 1] = score["overall"]
            for key, value in score.items():
                reward_metrics[key].append(value)

        ############################################
        # all_info_list.append(all_info)
        
        pred_point_tensor = torch.tensor(all_info_list_Pred_Point, dtype=torch.float32)
        fixed_bbox_list = fix_bbox_shape(all_info_list_GT_BBox_Ori)
        gt_bbox_tensor = torch.tensor(fixed_bbox_list, dtype=torch.float32)
        gt_point_tensor = torch.tensor(all_info_list_GT_Point, dtype=torch.float32)
        score_tensor = torch.tensor(all_info_list_Score, dtype=torch.float32)
        
        ############################################
        
        # return reward_tensor, reward_metrics
        return reward_tensor, reward_metrics, pred_point_tensor,gt_bbox_tensor,gt_point_tensor,score_tensor
